<?php

namespace App\Controllers;

use App\Models\ModeloModel;
use App\Models\MultimediaModel; // Importante añadir esto

class ModeloController extends BaseController
{
    protected $modeloModel;
    protected $multimediaModel;

    public function __construct() {
    $this->modeloModel = new ModeloModel();
    $this->multimediaModel = new MultimediaModel();
    $this->db = \Config\Database::connect();
    helper(['form', 'url', 'text', 'tiempo', 'email']);
}

    public function index() {
    $userId = session()->get('user_id');
    $db = \Config\Database::connect();
    
    // 1. Obtener los datos básicos de la modelo vinculada al usuario
    $modelo = $this->modeloModel->where('usuario_id', $userId)->first();

    // Si no tiene ficha, redirigir a crear/editar perfil
    if (!$modelo) {
        return redirect()->to(base_url('modelo/editar-ficha'));
    }

    // 2. Traer la galería multimedia de la modelo
    $galeria = $this->multimediaModel->where('modelo_id', $modelo['id'])->findAll();

    // 3. ESTADÍSTICAS: Contar Visitas Totales
    $vistas = $db->table('visitas_modelos')
                 ->where('modelo_id', $modelo['id'])
                 ->countAllResults();

    // 4. INTERACCIONES: Contar Clics de Contacto
    $clics = $db->table('clics_contacto')
                ->where('modelo_id', $modelo['id'])
                ->countAllResults();
                
    // 4.5. FAVORITOS: Contar cuántos usuarios la han guardado
    $favoritos = $db->table('favoritos')
                    ->where('modelo_id', $modelo['id'])
                    ->countAllResults();

    // 5. LÓGICA DE RANKINGS (Global y por Ciudad)
    
    // Subconsulta base para contar visitas por modelo
    $subqueryBase = $db->table('visitas_modelos')
                       ->select('modelo_id, COUNT(*) as total')
                       ->groupBy('modelo_id')
                       ->getCompiledSelect();
                       
     // 5.5. HISTORIAS ACTIVAS: ¿Tiene algo publicado en las últimas 24 horas?
    $historiasActivas = $db->table('modelo_historias') 
                       ->where('modelo_id', $modelo['id'])
                       ->where('creado_at >=', date('Y-m-d H:i:s', strtotime('-24 hours')))
                       ->countAllResults();

    // Ranking Global
    $posGlobal = $db->query("
        SELECT COUNT(*) + 1 as posicion 
        FROM ($subqueryBase) as stats 
        WHERE total > ?", [$vistas])->getRow()->posicion;

    // Ranking por Ciudad
    $subqueryCiudad = $db->table('visitas_modelos')
                         ->select('visitas_modelos.modelo_id, COUNT(*) as total')
                         ->join('modelos', 'modelos.id = visitas_modelos.modelo_id')
                         ->where('modelos.ciudad_id', $modelo['ciudad_id'])
                         ->groupBy('visitas_modelos.modelo_id')
                         ->getCompiledSelect();

    $posCiudad = $db->query("
        SELECT COUNT(*) + 1 as posicion 
        FROM ($subqueryCiudad) as stats 
        WHERE total > ?", [$vistas])->getRow()->posicion;

    // --- LÓGICA DE CONSEJOS INTELIGENTES ---
    $consejo = "¡Tu perfil está activo! Sigue así para atraer más clientes."; 
    $tipoConsejo = "info"; 
    
    if ($historiasActivas == 0) {
    $consejo = "No tienes historias activas. Las modelos con historias reciben hasta un 50% más de atención. ¡Sube una ahora!";
    $tipoConsejo = "primary"; // Color llamativo
    } 
    elseif ($vistas > 50 && ($clics / $vistas) < 0.05) {
        $consejo = "Tienes visitas pero pocos clics. Intenta mejorar tu foto principal.";
        $tipoConsejo = "warning";
    }
    
    elseif ($vistas > 50 && ($clics / $vistas) < 0.05) {
        $consejo = "Tienes muchas visitas pero pocos contactos. Prueba cambiando tu foto principal por una más llamativa.";
        $tipoConsejo = "warning";
    } 
    elseif ($posGlobal > 20 && $posCiudad > 5) {
        $consejo = "Estás perdiendo posiciones en tu ciudad. Sube una historia para volver a destacar.";
        $tipoConsejo = "primary";
    }
    elseif ($clics > 10 && ($clics / $vistas) > 0.15) {
        $consejo = "¡Excelente rendimiento! Tu perfil es muy efectivo. Mantén tus historias actualizadas.";
        $tipoConsejo = "success";
    }

    // 6. Preparar el array de datos para la vista
    $data = [
        'titulo'       => 'Mi Dashboard - Calientitas',
        'usuario'      => session()->get('nombre'),
        'modelo'       => $modelo,
        'galeria'      => $galeria,
        'vistas'       => $vistas,    
        'clics'        => $clics,     
        'favoritos'    => $favoritos,
        'pos_global'   => $posGlobal,
        'pos_ciudad'   => $posCiudad,
        'consejo'      => $consejo,      
        'tipo_consejo' => $tipoConsejo   
    ];

    return view('modelo/dashboard', $data);
}

    public function publicar() {
    $userId = session()->get('user_id');
    $db = \Config\Database::connect();
    
    // 1. Buscamos los datos básicos de la modelo
    $modelo = $this->modeloModel->where('usuario_id', $userId)->first();

    // --- NUEVO: Traer servicios maestros para el selector ---
    $data['servicios_maestros'] = $db->table('servicios_maestros')
                                     ->orderBy('nombre', 'ASC')
                                     ->get()->getResultArray();

    if (!$modelo) {
        $modelo = [
            'id'                  => null,
            'nombre_artistico'    => '',
            'edad'                => '',
            'telefono'            => '',
            'descripcion'         => '',
            'precio_clp'          => 0,
            'precio_oferta'       => 0,
            'ciudad_id'           => '',
            'horario_tipo'        => 'Full Time',
            'horario_detalle'     => '',
            'estatura'            => '',
            'medidas'             => '',
            'servicios_incluidos' => '',
            'servicios_adicionales' => '', // Aseguramos que exista
            'categoria'            => 'escort',
            'foto_principal'      => '',
            'lugar_atencion'      => '',
            'estacionamiento'      => 0,
            'medios_pago'          => '',
            'nacionalidad'         => '',
            'idiomas'              => '',
            'verificada'          => 0,
            'rut_verificacion'    => '',
            'selfie_verificacion' => '',
            'documento_identidad' => '',
            'observaciones_admin' => null,
        ];
    } else {
        $modelo['verificada'] = $modelo['verificada'] ?? 0;
        $modelo['documento_identidad'] = $modelo['documento_identidad'] ?? '';
        $modelo['selfie_verificacion'] = $modelo['selfie_verificacion'] ?? '';
    }

    $galeria = [];
    if ($modelo['id']) {
        $galeria = $this->multimediaModel->where('modelo_id', $modelo['id'])->findAll();
    }

    $data['regiones'] = $db->table('regiones')->orderBy('nombre', 'ASC')->get()->getResultArray();

    $data['region_actual'] = null;
    if (!empty($modelo['ciudad_id'])) {
        $ciudad = $db->table('ciudades')->where('id', $modelo['ciudad_id'])->get()->getRowArray();
        if ($ciudad) {
            $data['region_actual'] = $ciudad['region_id'];
        }
    }

    $data['titulo']  = 'Editar mi Perfil';
    $data['usuario'] = session()->get('nombre');
    $data['modelo']  = $modelo; 
    $data['galeria'] = $galeria;

    return view('modelo/publicar_ficha', $data);
}

    public function guardarFicha() {
    $userId = session()->get('user_id');
    $db = \Config\Database::connect(); 
    helper('email');
    
    $usuarioModel = new \App\Models\UsuarioModel();
    $usuario = $usuarioModel->find($userId);

    // 1. OBTENER EL RUT Y VALIDAR OBLIGATORIEDAD
    $rutVerificacion = $this->request->getPost('rut_verificacion');
    $existente = $this->modeloModel->where('usuario_id', $userId)->first();

    // Si la modelo no existe o no está verificada, el RUT es obligatorio
    if (!$existente || (isset($existente['verificada']) && $existente['verificada'] == 0)) {
        if (empty(trim($rutVerificacion))) {
            return redirect()->back()->withInput()->with('error', 'El RUT es obligatorio para solicitar la verificación.');
        }
    }

    // --- Procesar Servicios ---
    $incluidosArr = $this->request->getPost('servicios_incluidos'); 
    $adicionalesArr = $this->request->getPost('servicios_adicionales');

    $procesarServicios = function($servicios) use ($db) {
        if (empty($servicios)) return '';
        foreach ($servicios as $s) {
            $nombre = trim($s);
            if (!empty($nombre)) {
                $db->table('servicios_maestros')->ignore()->insert(['nombre' => $nombre]);
            }
        }
        return implode(',', $servicios);
    };

    // 2. AGREGAR RUT_VERIFICACION AL ARRAY DE DATOS
    $dataModelo = [
        'usuario_id'            => $userId,
        'nombre_artistico'      => $this->request->getPost('nombre_artistico'),
        'descripcion'           => $this->request->getPost('descripcion'),
        'edad'                  => $this->request->getPost('edad'),
        'estatura'              => $this->request->getPost('estatura'),
        'medidas'               => $this->request->getPost('medidas'),
        'telefono'              => $this->request->getPost('telefono'),
        'precio_clp'            => $this->request->getPost('precio_clp'),
        'categoria'             => $this->request->getPost('categoria'),
        'ciudad_id'             => $this->request->getPost('ciudad_id'),
        'rut_verificacion'      => $rutVerificacion, // <--- Guardamos el RUT aquí
        'servicios_incluidos'   => $procesarServicios($incluidosArr),
        'servicios_adicionales' => $procesarServicios($adicionalesArr),
        'precio_oferta'         => $this->request->getPost('precio_oferta') ?: null,
        'horario_tipo'          => $this->request->getPost('horario_tipo'),
        'horario_detalle'       => $this->request->getPost('horario_detalle'),
        'lugar_atencion'        => $this->request->getPost('lugar_atencion'),
        'estacionamiento'       => $this->request->getPost('estacionamiento'),
        'medios_pago'           => $this->request->getPost('medios_pago'),
        'activo'                => 1
    ];

    // --- SUBIDA DE DOCUMENTOS ---
    $subioDocumentos = false;
    $docFile = $this->request->getFile('documento_identidad');
    if ($docFile && $docFile->isValid() && !$docFile->hasMoved()) {
        $newName = 'rut_' . $userId . '_' . $docFile->getRandomName();
        $docFile->move(FCPATH . 'uploads/modelos/documentos', $newName);
        $dataModelo['documento_identidad'] = $newName;
        $dataModelo['verificada'] = 0;
        $subioDocumentos = true;
    }

    $selfieFile = $this->request->getFile('selfie_verificacion');
    if ($selfieFile && $selfieFile->isValid() && !$selfieFile->hasMoved()) {
        $newName = 'selfie_' . $userId . '_' . $selfieFile->getRandomName();
        $selfieFile->move(FCPATH . 'uploads/modelos/documentos', $newName);
        $dataModelo['selfie_verificacion'] = $newName;
        $dataModelo['verificada'] = 0;
        $subioDocumentos = true;
    }

    // FOTO PRINCIPAL PROCESADA
    $fotoPrincipal = $this->request->getFile('foto_principal');
    if ($fotoPrincipal && $fotoPrincipal->isValid()) {
        $nombreProcesado = $this->procesarImagenProtegida($fotoPrincipal);
        if ($nombreProcesado) {
            $dataModelo['foto_principal'] = $nombreProcesado;
        }
    }

    // 3. GUARDAR O ACTUALIZAR
    if ($existente) {
        $this->modeloModel->update($existente['id'], $dataModelo);
        $modeloId = $existente['id'];
    } else {
        $modeloId = $this->modeloModel->insert($dataModelo);
    }

    // GALERÍA PROCESADA
    $galeriaArchivos = $this->request->getFileMultiple('galeria');
    if ($galeriaArchivos) {
        foreach ($galeriaArchivos as $archivo) {
            if ($archivo && $archivo->isValid()) {
                $tipo = strpos($archivo->getMimeType(), 'video') !== false ? 'video' : 'foto';
                
                if ($tipo === 'video') {
                    $nombreVideo = $archivo->getRandomName();
                    $archivo->move(FCPATH . 'uploads/modelos/videos', $nombreVideo);
                    $nombreFinal = $nombreVideo;
                } else {
                    $nombreFinal = $this->procesarImagenProtegida($archivo, 'fotos');
                }

                if ($nombreFinal) {
                    $this->multimediaModel->insert([
                        'modelo_id' => $modeloId, 
                        'tipo' => $tipo, 
                        'ruta' => $nombreFinal, 
                        'es_principal' => 0
                    ]);
                }
            }
        }
    }

    if ($subioDocumentos) {
        $emailData = [
            'nombre' => $usuario['nombre'],
            'email'  => $usuario['email'],
            'id'     => $modeloId,
            'fecha'  => date('d-m-Y H:i')
        ];
        enviar_notificacion($usuario['email'], 'Documentos recibidos - En revisión 🔥', 'ficha_pendiente', $emailData);
        $adminEmail = 'kmarketexpresscl@gmail.com'; 
        enviar_notificacion($adminEmail, 'AVISO: Nueva Verificación de ' . $usuario['nombre'], 'admin_nueva_ficha', $emailData);
    }

    return redirect()->to(base_url('modelo/dashboard'))->with('success', 'Perfil actualizado correctamente.');
}
    
    public function eliminarMedia($id)
    {
    $userId = session()->get('user_id');
    
    // Buscar el archivo y verificar que pertenezca a la modelo logueada (Seguridad)
    $media = $this->multimediaModel
        ->select('modelo_multimedia.*')
        ->join('modelos', 'modelos.id = modelo_multimedia.modelo_id')
        ->where('modelo_multimedia.id', $id)
        ->where('modelos.usuario_id', $userId)
        ->first();

    if ($media) {
        // 1. Determinar la ruta del archivo
        $subCarpeta = ($media['tipo'] === 'video') ? 'videos' : 'fotos';
        $rutaFisica = FCPATH . 'uploads/modelos/' . $subCarpeta . '/' . $media['ruta'];

        // 2. Borrar archivo del servidor
        if (file_exists($rutaFisica)) {
            unlink($rutaFisica);
        }

        // 3. Borrar de la base de datos
        $this->multimediaModel->delete($id);

        return redirect()->back()->with('success', 'Archivo eliminado correctamente.');
    }

    return redirect()->back()->with('error', 'No se pudo eliminar el archivo o no tienes permisos.');
    }
    
    public function subirVerificacion() {
        die("ESTOY EDITANDO EL ARCHIVO CORRECTO");
        
    $userId = session()->get('user_id');
    $modelo = $this->modeloModel->where('usuario_id', $userId)->first();
    $usuarioModel = new \App\Models\UsuarioModel();
    $usuario = $usuarioModel->find($userId);

    // 1. Datos para la actualización
    $data = [
        'rut_verificacion'    => $this->request->getPost('rut_verificacion'),
        'verificada'          => 0,
        'observaciones_admin' => null 
    ];

    // 2. Procesar Archivos (A la ruta que confirmaste que funciona)
    $rutaDestino = FCPATH . 'uploads/modelos/documentos';

    $selfie = $this->request->getFile('selfie_verificacion');
    if ($selfie && $selfie->isValid()) {
        $nameS = 'selfie_' . $modelo['id'] . '_' . $selfie->getRandomName();
        $selfie->move($rutaDestino, $nameS);
        $data['selfie_verificacion'] = $nameS;
    }

    $rutImg = $this->request->getFile('documento_identidad');
    if ($rutImg && $rutImg->isValid()) {
        $nameR = 'rut_' . $modelo['id'] . '_' . $rutImg->getRandomName();
        $rutImg->move($rutaDestino, $nameR);
        $data['documento_identidad'] = $nameR;
    }

    // 3. ACTUALIZACIÓN CON DEBUGGER
    if ($this->modeloModel->update($modelo['id'], $data)) {
        
        // Si entra aquí, la base de datos SE ACTUALIZÓ.
        helper('email');

        $emailData = [
            'nombre'    => $usuario['nombre'],
            'email'     => $usuario['email'],
            'id_modelo' => $modelo['id'],
            'fecha'     => date('d-m-Y H:i')
        ];

        // FORZAMOS EL DEBUGGER DEL HELPER
        // Esto va a imprimir el log del correo y va a detener la ejecución (die)
        enviar_notificacion($usuario['email'], 'PRUEBA VERIFICACION', 'ficha_pendiente', $emailData);
        enviar_notificacion('hola@kmarketexpress.cl', 'PRUEBA ADMIN', 'admin_nueva_ficha', $emailData);

        return redirect()->to(base_url('modelo/dashboard'))->with('success', '¡Por fin! Archivos y correos procesados.');
        
    } else {
        // SI EL UPDATE FALLA, VEREMOS POR QUÉ AQUÍ:
        echo "<h1>FALLÓ EL UPDATE DEL MODELO</h1>";
        echo "<pre>";
        print_r($this->modeloModel->errors());
        echo "</pre>";
        die("Revisa si los campos en la DB aceptan NULL o si falta algún allowedField.");
    }
}

    public function actualizarDisponibilidad()
    {
        if (!$this->request->isAJAX()) {
            return $this->response->setStatusCode(403)->setJSON(['success' => false]);
        }
    
        $userId = session()->get('user_id');
        $json = $this->request->getJSON();
    
        if (!$json || !isset($json->disponible)) {
            return $this->response->setJSON(['success' => false, 'message' => 'Sin datos']);
        }
    
        $modeloModel = new \App\Models\ModeloModel();
        
        // Buscamos a la modelo por el ID de usuario
        $modelo = $modeloModel->where('usuario_id', $userId)->first();
    
        if (!$modelo) {
            return $this->response->setJSON(['success' => false, 'message' => 'Modelo no encontrada']);
        }
    
        // ACTUALIZACIÓN EXPLÍCITA: Pasamos el ID y el array de datos
        $datosActualizar = [
            'disponible_ahora' => (int)$json->disponible
        ];
    
        if ($modeloModel->update($modelo['id'], $datosActualizar)) {
            return $this->response->setJSON(['success' => true]);
        }
    
        return $this->response->setStatusCode(500)->setJSON(['success' => false]);
    }
    
    public function subirHistoria() {
    $userId = session()->get('user_id');
    $modeloModel = new \App\Models\ModeloModel();
    
    // 1. Buscamos el ID de la modelo
    $modelo = $modeloModel->where('usuario_id', $userId)->first();
    if (!$modelo) return redirect()->back()->with('error', 'Perfil no encontrado');

    // 2. Procesamos el archivo de la historia
    $archivo = $this->request->getFile('foto_historia');

    if ($archivo && $archivo->isValid() && !$archivo->hasMoved()) {
        
        $tipo = strpos($archivo->getMimeType(), 'video') !== false ? 'video' : 'foto';
        
        if ($tipo === 'foto') {
            // --- OPTIMIZACIÓN DE FOTO ---
            // Usamos tu procesador para bajar peso, calidad y redimensionar
            // Esto hará que la historia cargue súper rápido
            $nombreArchivo = $this->procesarImagenProtegida($archivo, 'historias');
        } else {
            // --- VALIDACIÓN DE VIDEO ---
            // Validamos que el video no supere los 15MB para no saturar el servidor
            if ($archivo->getSizeByUnit('mb') > 15) {
                return redirect()->back()->with('error', 'El video es demasiado pesado. Máximo 15MB.');
            }
            $nombreArchivo = $archivo->getRandomName();
            $archivo->move(FCPATH . 'uploads/modelos/historias', $nombreArchivo);
        }

        if ($nombreArchivo) {
            // 3. Guardamos en la base de datos con la expiración de 24 horas
            $db = \Config\Database::connect();
            $dataHistoria = [
                'modelo_id' => $modelo['id'],
                'ruta'      => $nombreArchivo,
                'creado_at' => date('Y-m-d H:i:s'),
                'expira_at' => date('Y-m-d H:i:s', strtotime('+24 hours'))
            ];

            $db->table('modelo_historias')->insert($dataHistoria);

            return redirect()->back()->with('success', '¡Historia publicada por 24 horas!');
        }
    }

    return redirect()->back()->with('error', 'No se pudo subir el archivo multimedia.');
}
    
        public function verHistorias() {
        $userId = session()->get('user_id');
    
        if (!$userId) {
            return redirect()->to(base_url('login'));
        }
    
        $modelo = $this->modeloModel->where('usuario_id', $userId)->first();
    
        if (!$modelo) {
            return redirect()->to(base_url('modelo/editar-ficha'))
                             ->with('error', 'Crea tu perfil antes de subir historias.');
        }
    
        $historias = $this->db->table('modelo_historias')
            ->where('modelo_id', $modelo['id'])
            ->where('expira_at >', date('Y-m-d H:i:s'))
            ->orderBy('creado_at', 'DESC')
            ->get()
            ->getResultArray();
    
        // AQUÍ ESTÁ EL FIX: Pasamos 'usuario' a la vista
        $data = [
            'modelo'    => $modelo,
            'usuario'   => $modelo['nombre_artistico'], // Esto quita el error del Undefined Variable
            'historias' => $historias,
            'titulo'    => 'Subir Historias - Calientitas'
        ];
    
        return view('modelo/historias', $data);
    }
        public function eliminarHistoria($id) {
        $userId = session()->get('user_id');
        $modelo = $this->modeloModel->where('usuario_id', $userId)->first();
    
        // Verificamos que la historia pertenezca a la modelo para seguridad
        $historia = $this->db->table('modelo_historias')
            ->where('id', $id)
            ->where('modelo_id', $modelo['id'])
            ->get()->getRow();
    
        if ($historia) {
            // Borramos el archivo físico
            $rutaFisica = FCPATH . 'uploads/modelos/historias/' . $historia->ruta;
            if (file_exists($rutaFisica)) unlink($rutaFisica);
    
            // Borramos de la BD
            $this->db->table('modelo_historias')->where('id', $id)->delete();
    
            return redirect()->back()->with('success', 'Historia eliminada correctamente.');
        }
    
        return redirect()->back()->with('error', 'No se pudo eliminar la historia.');
    }
        public function planes() {
        $userId = session()->get('user_id');
        
        // Buscamos la modelo real
        $modelo = $this->modeloModel->where('usuario_id', $userId)->first();
    
        // VALIDACIÓN DE SEGURIDAD:
        // Si no existe la ficha en la DB, la mandamos a editar-ficha
        if (!$modelo) {
            return redirect()->to(base_url('modelo/editar-ficha'))
                             ->with('info', 'Debes crear tu ficha antes de elegir un plan.');
        }
    
        // Si llegamos aquí, es porque la modelo SÍ existe
        $data = [
            'titulo'  => 'Elegir Plan',
            'usuario' => session()->get('nombre') ?? 'Modelo',
            'modelo'  => $modelo
        ];
    
        return view('modelo/planes', $data); 
    }
    
    // Función para proteger y optimizar imágenes (Pegar al final del controlador)
    private function procesarImagenProtegida($file, $subCarpeta = '') {
        if ($file && $file->isValid() && !$file->hasMoved()) {
            
            $newName = $file->getRandomName();
            $tempPath = WRITEPATH . 'uploads/' . $newName; 
            $destinoFinal = FCPATH . 'uploads/modelos/' . ($subCarpeta ? $subCarpeta . '/' : '');
            $finalPath = $destinoFinal . $newName;
    
            $file->move(WRITEPATH . 'uploads', $newName);
    
            $image = \Config\Services::image();
    
            // --- SOLUCIÓN ORIENTACIÓN Y OPTIMIZACIÓN ---
            $image->withFile($tempPath)
                  ->reorient() // <--- ESTO corrige las fotos que se suben "acostadas"
                  ->resize(1200, 1200, true, 'auto') // 'auto' decide mejor según la foto
                  ->save($tempPath, 80); 
    
            // --- MARCA DE AGUA MÁS GRANDE Y DIAGONAL ---
            $imgProcesada = $image->withFile($tempPath);
    
            // Definimos posiciones estratégicas
            $posiciones = [
                ['h' => 'center', 'v' => 'middle'],
                ['h' => 'left',   'v' => 'top'],
                ['h' => 'right',  'v' => 'bottom'],
            ];
    
            foreach ($posiciones as $pos) {
                $imgProcesada->text('CALIENTITAS.CL', [
                    'color'      => '#ffffff',
                    'opacity'    => 0.2,        // Un poco más transparente para compensar el tamaño
                    'withShadow' => true,
                    'hAlign'     => $pos['h'],
                    'vAlign'     => $pos['v'],
                    'fontSize'   => 45,         // <--- MÁS GRANDE
                    'padding'    => 50,
                    // Nota: CodeIgniter nativo no tiene 'rotation' en el método text(), 
                    // pero si usas ImageMagick en el servidor se puede. 
                    // Por defecto simularemos la diagonal con la posición de las esquinas.
                ]);
            }
    
            $imgProcesada->save($finalPath);
            if (file_exists($tempPath)) unlink($tempPath);
    
            return $newName;
        }
        return null;
    }

}